# -*- coding: utf-8 -*-
"""
/***************************************************************************
 CDVI
                                 A QGIS plugin
 City Disaster Vulnerability Index
                             -------------------
        begin                : 2020-06-11
        copyright            : (C) 2020 by WB
        email                : andlang@outlook.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os

from cdvi.ui.about_dialog import AboutDialog
from cdvi.ui.add_indicators import AddIndicatorsDialog
from cdvi.utilities.city_data_loader import CityDataLoader
from cdvi.utilities.data_loader import DataLoader
from cdvi.utilities.utils import create_item
from qgis.PyQt import QtWidgets
from qgis.PyQt import uic
from qgis.PyQt.QtCore import pyqtSlot, pyqtSignal, QObject, QEvent
from qgis.PyQt.QtGui import QStandardItemModel
from qgis.PyQt.QtWidgets import QHeaderView

# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer

FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'cdvi_dialog_base.ui'))


class CDVIDialog(QtWidgets.QDialog, FORM_CLASS):
    def __init__(self, plugin_dir: str, parent=None):
        self.plugin_dir = plugin_dir
        self.setup_options = DataLoader(self.plugin_dir).get_setup_options()
        self.cities = DataLoader(self.plugin_dir).get_cities()

        self.additional_indicators_pillar = ''
        self.additional_indicators = []

        """Constructor."""
        super(CDVIDialog, self).__init__(parent)
        # Set up the user interface from Designer through FORM_CLASS.
        # After self.setupUi() you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect
        self.setupUi(self)

        self.logos_label.setOpenExternalLinks(True)

        clickable(self.logos_label).connect(self.on_logos_label_clicked)

        self.city_cbx.clear()
        self.city_cbx.addItems([city[0] for city in self.cities])
        self.city_cbx.currentIndexChanged.connect(self.city_selection_changed)

        self.sector_cbx.clear()
        self.sector_cbx.addItems([sector for sector in self.setup_options['Sector_Field'].unique()])
        self.sector_cbx.currentIndexChanged.connect(self.sector_selection_changed)

        self.context_cbx.clear()
        self.context_cbx.addItems(
            [context for context in self.setup_options.query('Sector_Code == 0')['Context_Field']])

        self.change_indicator_details()

    def get_selected_city(self):
        return self.cities[self.city_cbx.currentIndex()]

    def get_initial_data_matrix(self):
        selected_city = self.get_selected_city()
        selected_sector = str(self.sector_cbx.currentIndex())
        return CityDataLoader(self.plugin_dir, selected_city[1], selected_city[0],
                              selected_city[2]).get_initial_data_matrix().query(
            'Sector.str.startswith(@selected_sector)')

    def city_selection_changed(self, i):
        self.change_indicator_details()

    def sector_selection_changed(self, i):
        self.context_cbx.clear()
        self.context_cbx.addItems(
            [context for context in self.setup_options.query('Sector_Code == @i')['Context_Field']])
        self.change_indicator_details()

    def change_indicator_details(self):
        model = QStandardItemModel()
        model.setColumnCount(3)

        for _, row in self.get_initial_data_matrix().iterrows():
            model.appendRow(
                [create_item(row['Indicator']), create_item(row['Pillar']), create_item(row['Description'])])

        self.indicator_details.setModel(model)
        self.indicator_details.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        self.indicator_details.horizontalHeader().hide()
        self.indicator_details.verticalHeader().hide()

    @pyqtSlot()
    def on_get_indicators_btn_clicked(self):
        dlg = AddIndicatorsDialog(self.plugin_dir, self.get_selected_city(),
                                  self.additional_indicators_pillar, self.additional_indicators)
        dlg.show()
        # Run the dialog event loop
        result = dlg.exec_()
        # See if OK was pressed
        if result:
            self.additional_indicators_pillar = dlg.selected_pillar
            self.additional_indicators = dlg.indicator_multiselect.get_selected_items()
            pass

    def on_logos_label_clicked(self):
        dlg = AboutDialog()
        dlg.show()
        # Run the dialog event loop
        result = dlg.exec_()
        # See if OK was pressed
        if result:
            pass


def clickable(widget):
    class Filter(QObject):

        clicked = pyqtSignal()

        def eventFilter(self, obj, event):

            if obj == widget:
                if event.type() == QEvent.MouseButtonRelease:
                    if obj.rect().contains(event.pos()):
                        self.clicked.emit()
                        # The developer can opt for .emit(obj) to get the object within the slot.
                        return True

            return False

    filter = Filter(widget)
    widget.installEventFilter(filter)
    return filter.clicked
