const formatInputData = (result, parentID, parentWeight, item) => {
  result.push({
    ids: item.id,
    labels: item.name,
    parents: parentID,
    values: item.weight * parentWeight
  });

  let sum = item.children.map(it => it.weight).reduce((a, b) => a + b, 0);
  let divisor = sum > 1 ? sum : 1;

  item.children.forEach(child => {
    return formatInputData(
      result,
      item.id,
      (item.weight * parentWeight) / divisor,
      child
    );
  });

  return result;
};

const unpack = (rows, key) => {
  return rows.map(row => row[key]);
};

const json2Table = (json, changeable) => {
  let cols = Object.keys(json[0]);

  let headerRow = cols
    .slice(1, 3)
    .map(col => `<th>${col}</th>`)
    .join("");

  let rows = json
    .map(row => {
      let id = row[cols[0]];
      let value = Math.round((row[cols[2]] + Number.EPSILON) * 100000) / 100000;

      let nameCell = changeable
        ? `<label for="${id}">${row[cols[1]]}</label>`
        : row[cols[1]];
      let valueCell = changeable
        ? `<input type="text" id="${id}" value="${value}">`
        : value;

      return `
        <tr>
          <td>${nameCell}</td>
          <td>${valueCell}</td>
        </tr>`;
    })
    .join("");

  return `
    <table id="weight_table">
      <thead>
      	<tr>${headerRow}</tr>
      <thead>
      <tbody>
      	${rows}
      <tbody>
    <table>`;
};

const getChildren = (json, array) => {
  if (!json) {
    return "";
  }
  if (array.length) {
    var element = array.shift();
    return getChildren(
      json.children.find(el => el.name == element),
      array
    );
  }

  return json.children;
};

const updateValue = (json, id, weight) => {
  let element = find(json.children, id);
  if (element) {
    element.weight = weight;
  }
};

function find(source, id) {
  for (key in source) {
    var item = source[key];
    if (item.id == id) {
      return item;
    }

    if (item.children) {
      var subresult = find(item.children, id);

      if (subresult) {
        return subresult;
      }
    }
  }
  return null;
}

const loadPD = qt_page => {
  var inputJSON = JSON.parse(qt_page.json_str);
  var inputData = formatInputData([], "", 1, inputJSON);
  var selectedChildren;

  if (!qt_page.CHANGEABLE) {
    document.querySelector("#update-button").style.display = "none";
  }

  const updateTable = () => {
    document.querySelector("#table").innerHTML = json2Table(
      selectedChildren ? selectedChildren : inputJSON.children,
      qt_page.CHANGEABLE
    );
  };

  const updateGraph = () => {
    inputData = formatInputData([], "", 1, inputJSON);
    graph.data[0].values = unpack(inputData, "values");
    Plotly.redraw(graph);
  };

  updateTable();

  document.querySelector("#update-button").onclick = () => {
    let inputs = Array.from(
      document.querySelectorAll("#weight_table input[type=text]")
    );

    let sum = inputs.map(it => parseFloat(it.value)).reduce((a, b) => a + b, 0);

    inputs.forEach(it => {
      updateValue(inputJSON, it.id, parseFloat(it.value) / sum);
    });

    updateTable();
    updateGraph();
    qt_page.json_updated(inputJSON);
  };

  var data = [
    {
      type: "sunburst",
      ids: unpack(inputData, "ids"),
      labels: unpack(inputData, "labels"),
      parents: unpack(inputData, "parents"),
      values: unpack(inputData, "values"),
      outsidetextfont: { size: 20, color: "#377eb8" },
      leaf: { opacity: 0.4 },
      marker: { line: { width: 2 } },
      branchvalues: "total",
      textposition: "inside",
      insidetextorientation: "radial"
    }
  ];

  var layout = {
    margin: { l: 0, r: 0, b: 0, t: 0 },
    width: 500,
    height: 500,
    hovermode: false
  };

  Plotly.newPlot(graph, data, layout, { displayModeBar: false }).then(gd => {
    gd.on("plotly_sunburstclick", event => {
      if (event.nextLevel) {
        var splitted = event.nextLevel.split(".").slice(1);
        selectedChildren = getChildren(inputJSON, splitted);

        updateTable();
      }
    });
  });
};
