# -*- coding: utf-8 -*-
"""
/***************************************************************************
 CDVI
                                 A QGIS plugin
 City Disaster Vulnerability Index
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2020-06-11
        git sha              : $Format:%H$
        copyright            : (C) 2020 by WB
        email                : andlang@outlook.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import json
import os.path
import uuid
from copy import deepcopy
from typing import List

# DO NOT REMOVE THIS
# noinspection PyUnresolvedReferences
import cdvi.resources  # pylint: disable=unused-import  # NOQA
import numpy as np
import pandas as pd
import processing
from cdvi.ui.calculate_cdvi_dialog import CalculateCDVIDialog
from cdvi.ui.cdvi_dialog import CDVIDialog
from cdvi.ui.help_dialog import HelpDialog
from cdvi.ui.no_layer_selected_dialog import NoLayerSelectedDialog
from cdvi.ui.weight_data_dialog import WeightDataDialog
from cdvi.ui.what_if_dialog import WhatIfDialog
from cdvi.utilities.city_data_loader import CityDataLoader
from cdvi.utilities.data_loader import DataLoader
from cdvi.utilities.utils import write_layer_suppl_info_to_qgs, get_layer_suppl_info_from_qgs
from cdvi.utilities.weight_data_loader import WeightDataLoader
from cdvi.utilities.weight_data_loader import WeightItem
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QVariant
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog, QDialog, QToolButton, QMenu
from qgis.core import QgsVectorLayer, QgsMapLayer, QgsProject, QgsField, QgsSymbol, QgsRendererCategory, \
    QgsCategorizedSymbolRenderer, QgsStyle, QgsPointXY


class CDVI:

    def __init__(self, iface):

        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.setup_options = DataLoader(self.plugin_dir).get_setup_options()

        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'CDVI_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        self.actions = []
        self.menu = self.tr(u'&City Disaster Vulnerability Index Tool')
        self.popupMenu = QMenu(self.iface.mainWindow())

        self.first_start = None

        self.destination = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('CDVI', message)

    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.popupMenu.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        self.add_action(
            ':/plugins/cdvi/start.svg',
            text=self.tr(u'City Disaster Vulnerability Index Tool'),
            callback=self.run,
            parent=self.iface.mainWindow())

        self.add_action(
            ':/plugins/cdvi/calculate.svg',
            text=self.tr(u'Calculate Vulnerability'),
            callback=self.on_calculate_vulnerability_clicked,
            parent=self.iface.mainWindow())

        self.add_action(
            ':/plugins/cdvi/weights.svg',
            text=self.tr(u'Change Weights'),
            callback=self.on_change_weights_clicked,
            parent=self.iface.mainWindow())

        self.add_action(
            ':/plugins/cdvi/whatIf.svg',
            text=self.tr(u'What-if analysis'),
            callback=self.on_what_if_analysis_clicked,
            parent=self.iface.mainWindow())

        self.add_action(
            ':/plugins/cdvi/help.svg',
            text=self.tr(u'Show Help'),
            callback=self.on_help_clicked,
            parent=self.iface.mainWindow())

        tool_button = QToolButton()

        tool_button.setMenu(self.popupMenu)
        tool_button.setDefaultAction(self.actions[0])
        tool_button.setPopupMode(QToolButton.InstantPopup)

        self.tb_action = self.iface.addToolBarWidget(tool_button)

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&City Disaster Vulnerability Index Tool'),
                action)
        self.iface.removeToolBarIcon(self.tb_action)

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start:
            self.first_start = False
            self.dlg = CDVIDialog(self.plugin_dir)

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            selected_city = self.dlg.get_selected_city()
            city_dir = CityDataLoader(self.plugin_dir, selected_city[1], selected_city[0], selected_city[2]).city_dir

            for _, row in self.dlg.get_initial_data_matrix().iterrows():
                self.add_layer_for_row(city_dir, row, selected_city)

            if len(self.dlg.additional_indicators) > 0:
                self.add_additional_indicators_layer(selected_city)

    def add_layer_for_row(self, city_dir: str, row: dict, selected_city: List[str]):
        input_layer = QgsVectorLayer(
            city_dir + "Initial_Data/" + row['Sector'] + "/" + row['Shapefile'], '', "ogr")
        output_layer = self.copy_and_save_layer(self.dlg, input_layer,
                                                self.get_selected_sector(self.dlg.sector_cbx.currentIndex()) + " - " +
                                                row['Indicator'], selected_city[2])

        self.add_suppl_info(output_layer, selected_city)
        self.style_categorized(output_layer)
        QgsProject.instance().addMapLayer(output_layer)

    def add_suppl_info(self, layer: QgsMapLayer, selected_city: List[str]):
        suppl_info = {'country_abbr': selected_city[1],
                      'city_name': selected_city[0],
                      'city_abbr': selected_city[2],
                      'sector': self.dlg.sector_cbx.currentIndex(),
                      'context': self.dlg.context_cbx.currentIndex()}
        write_layer_suppl_info_to_qgs(layer.id(), suppl_info)

    def add_to_suppl_info(self, layer: QgsMapLayer, suppl_info: dict, weights: WeightItem, multipliers: pd.DataFrame):
        suppl_info_copy = deepcopy(suppl_info)

        suppl_info_copy['weights'] = json.dumps(weights, default=lambda obj: obj.__dict__)
        suppl_info_copy['multipliers'] = multipliers.to_json(orient='split')

        write_layer_suppl_info_to_qgs(layer.id(), suppl_info_copy)

    def add_additional_indicators_layer(self, selected_city: List[str]):
        additional_layer = self.copy_and_save_layer(self.dlg, self.get_selected_layer(),
                                                    self.dlg.additional_indicators_pillar + '_Added_Indicators',
                                                    selected_city[2])
        additional_data = CityDataLoader(self.plugin_dir, selected_city[1], selected_city[0],
                                         selected_city[2]) \
            .combine_data(self.dlg.additional_indicators_pillar) \
            .filter(items=self.dlg.additional_indicators)

        self.add_suppl_info(additional_layer, selected_city)
        self.add_features(additional_layer, additional_data)
        QgsProject.instance().addMapLayer(additional_layer)

    def style_categorized(self, layer: QgsMapLayer):
        field_name = layer.fields().names()[-1]
        field_index = layer.fields().indexFromName(field_name)
        categories = []

        for unique_value in layer.uniqueValues(field_index):
            symbol = QgsSymbol.defaultSymbol(layer.geometryType())
            category = QgsRendererCategory(unique_value, symbol, str(unique_value))
            categories.append(category)
        renderer = QgsCategorizedSymbolRenderer(field_name, categories)
        renderer.sortByValue()
        style = QgsStyle().defaultStyle()
        ramp = style.colorRamp('Greens')
        renderer.updateColorRamp(ramp)

        if renderer is not None:
            layer.setRenderer(renderer)
        layer.triggerRepaint()

        self.iface.mapCanvas().refresh()

    def copy_and_save_layer(self, dlg: QDialog, input_layer: QgsMapLayer, indicator: str,
                            city_abbr: str) -> QgsMapLayer:
        if not self.destination:
            self.destination = self.ask_for_destination_folder(dlg)

        input_layer.selectAll()
        output = processing.run("native:saveselectedfeatures",
                                {'INPUT': input_layer,
                                 'OUTPUT': self.destination + '/' + city_abbr + '_' + "".join(
                                     x for x in indicator if (x.isalnum() or x in "._- ")) + '_' + str(
                                     uuid.uuid4()) + '.shp'})[
            'OUTPUT']
        output_layer = QgsVectorLayer(output, indicator, "ogr")

        return output_layer

    def calculate_index(self, weight_item: WeightItem, df: pd.DataFrame, composite_data: pd.DataFrame,
                        prefix: str) -> pd.DataFrame:
        if weight_item.level == "3.0":
            result = sum(child.weight * df[child.name] for child in weight_item.children if child.name in df)
            composite_data[prefix + weight_item.name] = result
            return result

        result = sum(
            self.calculate_index(child, df, composite_data, weight_item.name[:2]) for child in weight_item.children)
        composite_data[prefix + weight_item.name] = result
        return result

    def on_calculate_vulnerability_clicked(self):
        selected_layer = self.get_selected_layer()
        if not selected_layer:
            return

        dlg = CalculateCDVIDialog(self.plugin_dir, selected_layer)
        dlg.show()
        # Run the dialog event loop
        result = dlg.exec_()
        # See if OK was pressed
        if result:
            suppl_info = get_layer_suppl_info_from_qgs(selected_layer.id())
            self.calculate_and_add_vulnerability_layer(dlg, selected_layer, suppl_info)

    def calculate_and_add_vulnerability_layer(self, dlg: QDialog, selected_layer: QgsMapLayer, suppl_info: dict,
                                              prefix: str = "",
                                              weight_data: WeightItem = None,
                                              multipliers: pd.DataFrame = pd.DataFrame()):
        if not weight_data:
            weight_data = self.get_weight_data(suppl_info)

        if multipliers.empty:
            multipliers = self.get_multipliers(suppl_info)

        composite_data = CityDataLoader(self.plugin_dir, suppl_info['country_abbr'], suppl_info['city_name'],
                                        suppl_info['city_abbr']).combine_data_and_apply_transformation('Desc',
                                                                                                       multipliers)
        composite_data['CDVI'] = sum(
            child.weight * self.calculate_composite_data(weight_data, child, dlg, composite_data, selected_layer,
                                                         suppl_info, prefix, multipliers) for
            child in weight_data.children)
        layer = self.copy_and_save_layer(dlg, selected_layer,
                                         prefix + self.get_selected_sector(suppl_info['sector']) + ' - CDVI Composite',
                                         suppl_info['city_abbr'])
        self.add_features(layer, composite_data)
        self.style_categorized(layer)
        self.add_to_suppl_info(layer, suppl_info, weight_data, multipliers)

        QgsProject.instance().addMapLayer(layer)

    def calculate_composite_data(self, weight_data: WeightItem, weight_item: WeightItem, dlg: QDialog,
                                 composite_data: pd.DataFrame, selected_layer: QgsMapLayer,
                                 suppl_info: dict, prefix: str, multipliers: pd.DataFrame) -> pd.DataFrame:
        layer = self.copy_and_save_layer(dlg, selected_layer, prefix + self.get_selected_sector(
            suppl_info['sector']) + " - " + weight_item.name, suppl_info['city_abbr'])

        data = CityDataLoader(self.plugin_dir, suppl_info['country_abbr'], suppl_info['city_name'],
                              suppl_info['city_abbr']).combine_data_and_apply_transformation(weight_item.name,
                                                                                             multipliers)

        self.add_features(layer, data)
        self.style_categorized(layer)
        self.add_to_suppl_info(layer, suppl_info, weight_data, multipliers)

        QgsProject.instance().addMapLayer(layer)

        return self.calculate_index(weight_item, data, composite_data, 'PI_')

    def get_selected_sector(self, i: int) -> str:
        return self.setup_options.query('Sector_Code == @i').iloc[0]['Sector_Field']

    def add_features(self, layer: QgsMapLayer, data: pd.DataFrame):
        attributes = []
        layer_provider = layer.dataProvider()

        layer_provider.deleteAttributes(layer_provider.attributeIndexes())
        layer.updateFields()

        dtypes = data.infer_objects().dtypes

        for col in data.columns:
            attr = QgsField(col if len(col) < 11 else col[:4] + '_' + col[-5:], self.get_feature_type(dtypes[col]))
            attributes.append(attr)

        layer.startEditing()
        layer_provider.addAttributes(attributes)
        layer.updateFields()

        for index, col in enumerate(data.columns):
            layer.setFieldAlias(index, col)

        for feature in layer.getFeatures():
            feature_id = feature.id()
            row = data.loc[feature_id]
            [feature.setAttribute(x if len(x) < 11 else x[:4] + '_' + x[-5:], row[x]) for x in row.index]
            layer.updateFeature(feature)

        layer.commitChanges()
        layer.triggerRepaint()

    def get_feature_type(self, field_type) -> QVariant:
        if field_type == np.int64:
            return QVariant.Int
        if field_type == np.float64:
            return QVariant.Double
        return QVariant.String

    def ask_for_destination_folder(self, parent: QDialog):
        return QFileDialog.getExistingDirectory(parent, 'Destination', os.path.expanduser("~"))

    def on_change_weights_clicked(self):
        selected_layer = self.get_selected_layer()
        if not selected_layer:
            return
        suppl_info = get_layer_suppl_info_from_qgs(selected_layer.id())

        weight_data = self.get_weight_data(suppl_info)

        dlg = WeightDataDialog(weight_data, True)
        dlg.show()
        # Run the dialog event loop
        result = dlg.exec_()
        # See if OK was pressed
        if result:
            if dlg.modified_weights:
                self.calculate_and_add_vulnerability_layer(dlg, selected_layer, suppl_info, "Changed Weights ",
                                                           weight_data=WeightItem.from_dict(dlg.modified_weights))
            else:
                self.calculate_and_add_vulnerability_layer(dlg, selected_layer, suppl_info, weight_data=weight_data)

    def get_weight_data(self, suppl_info) -> WeightItem:
        return WeightItem.from_dict(json.loads(suppl_info['weights'])) if 'weights' in suppl_info else WeightDataLoader(
            self.plugin_dir, suppl_info['country_abbr'], suppl_info['city_name'],
            suppl_info['city_abbr']).get_weight_data(
            str(suppl_info['sector']) + '_' + str(suppl_info['context']))

    def get_multipliers(self, suppl_info) -> pd.DataFrame:
        return pd.read_json(suppl_info['multipliers'],
                            orient='split') if 'multipliers' in suppl_info else CityDataLoader(
            self.plugin_dir, suppl_info['country_abbr'], suppl_info['city_name'],
            suppl_info['city_abbr']).get_multipliers()

    def on_what_if_analysis_clicked(self):
        selected_layer = self.get_selected_layer()
        if not selected_layer:
            return
        suppl_info = get_layer_suppl_info_from_qgs(selected_layer.id())

        multipliers = self.get_multipliers(suppl_info)

        dlg = WhatIfDialog(multipliers)
        dlg.show()
        # Run the dialog event loop
        result = dlg.exec_()
        # See if OK was pressed
        if result:
            self.calculate_and_add_vulnerability_layer(dlg, selected_layer, suppl_info, "WhatIf ",
                                                       weight_data=self.get_weight_data(suppl_info),
                                                       multipliers=multipliers)

    def get_selected_layer(self) -> QgsMapLayer:
        selected_layers = self.iface.layerTreeView().selectedLayers()

        if not selected_layers:
            dlg = NoLayerSelectedDialog()
            dlg.show()
            dlg.exec_()
            return

        return selected_layers[0]

    def on_help_clicked(self):
        dlg = HelpDialog()
        dlg.show()
        dlg.exec_()
        return
